/*
 * Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 * SPDX-License-Identifier:  GPL-2.0+
 */

#include <common.h>
#include <malloc.h>
#include "unlocker.h"
#include "unlocker_impl.h"
#include "ring_platform_features.h"
#ifdef CONFIG_MS_SPINAND
#include "drvSPINAND.h"
#include "spinand.h"
#endif
#include "asm/arch/mach/platform.h"
#include "drvAESDMA.h"


extern int read_nand_partition_from_type(int part_type, size_t size_to_read, u_char *addr, int dev);
extern int erase_and_write_nand_partition_to_type(int part_type, size_t size_to_write, u_char *addr, int dev);
extern unsigned int get_ms_ticks(void);

#ifdef CONFIG_CMD_OTPCTRL // SAV533 usage case
#define UNINITIALIZED_VAL       0xff
#define SECUREBOOT_COMMAND      0x02
extern void otp_readback(unsigned long  otp_cmd, u8* data, unsigned long dataSize);
#endif
bootloader_status_t device_production_locked(bool * secured_status_out) {
#ifdef CONFIG_CMD_OTPCTRL
	u8 secureboot_val = UNINITIALIZED_VAL;
	unsigned long otp_cmd = SECUREBOOT_COMMAND;
	otp_readback(otp_cmd, &secureboot_val, sizeof(secureboot_val));
	if (secureboot_val == UNINITIALIZED_VAL) {
		return BOOTLOADER_READ_ERROR;
	}
	if (secureboot_val) {
		*secured_status_out = true;
	} else {
		*secured_status_out = false;
	}
#else
// This only applies to SAV526.  SAV533 uses device OTP
#if (defined ALWAYS_PROD_LOCKED) && (ALWAYS_PROD_LOCKED == 0)
	*secured_status_out = MDrv_SPINAND_IsOtpLocked();
#else
	*secured_status_out = TRUE;
#endif
#endif
	return BOOTLOADER_OK;
}

#define FACTORY_TEST_ENABLED 0xFAC77E57

typedef struct {
    uint32_t    enabled;  /* enabled if set to FACTORY_TEST_ENABLED */
    uint32_t    port;     /* port for test code connect */
} factory_test_data_t;

struct unlock_meta {
	unlock_block_t unlock_block;
	uint32_t boot_counter;

    /* NON-UNLOCK ITEMS.  These are here as this is the only block of flash we have
       reserved which will never get manually reflashed and will persist across reboots.
       Since we have to erase the whole block at write, it is easier to manage all variables
       together */
	factory_test_data_t factory_test;
};

static unique_tag_t dsn;
static bool dsn_valid;

/* Static variable to store unlock state during a boot.  Not stored, the state
   should be re-calculated per boot */
static unlock_state_t static_unlock_state = UNLOCK_STATE_INVALID;

/*
 * Store the current unlock state.  This should _not_ be stored in non-volatile memory
 * as it should be recalculated per boot
 *
 * @return BOOTLOADER_OK if successfully stored otherwise BOOTLOADER_INVALID_PARAMETER
 *  if not a valid state, or other error as appropriate
 */
bootloader_status_t platform_set_unlock_state(unlock_state_t unlock_state)
{
	bootloader_status_t ret = BOOTLOADER_INVALID_PARAMETER;
	if ((unlock_state >= UNLOCK_STATE_LOCKED) &&
	    (unlock_state <= UNLOCK_STATE_NON_PROD)) {
		static_unlock_state = unlock_state;
		ret = BOOTLOADER_OK;
	} else {
		static_unlock_state = UNLOCK_STATE_INVALID;
	}
	return ret;
}

/*
 * Retrieve the current unlock state.  This should _not_ be stored in non-volatile memory
 * as it should be recalculated per boot
 *
 * @return BOOTLOADER_OK if successfully retrieved otherwise BOOTLOADER_INVALID_PARAMETER
 *  if a NULL pointer
 */
bootloader_status_t platform_get_unlock_state(unlock_state_t *unlock_state)
{
	bootloader_status_t ret = BOOTLOADER_INVALID_PARAMETER;
	if (unlock_state) {
		*unlock_state = static_unlock_state;
		ret = BOOTLOADER_OK;
	}
	return ret;
}

/*
 * Secure boot enabled, JTAG disabled, OTP locked, Bootrom download disabled etc.
 * @return BOOTLOADER_OK if successfully retrieved otherwise BOOTLOADER_PARTIALLY_SECURED
 *  or BOOTLOADER_READ_ERROR
 */
bootloader_status_t platform_get_secured_status(bool * secured_status_out)
{
	bootloader_status_t ret = BOOTLOADER_INVALID_PARAMETER;
	if (secured_status_out) {
		// SAV533 - check secureboot value
		// SAV526 - implement original logic
		ret = device_production_locked(secured_status_out);
		if (ret != BOOTLOADER_OK) {
			return ret;
		}
		if (*secured_status_out == false) {
			/* We're not prod locked but we might want to simulate
			   locked state. This only simulates for U-Boot, not
			   RTOS */
			if (getenv_yesno("simulate_locked") == 1)
				*secured_status_out = true;
		}
		ret = BOOTLOADER_OK;
	}
	return ret;
}

/*
 * Get unlock block from storage location
 * @return BOOTLOADER_OK if successfully retrieved otherwise BOOTLOADER_READ_ERROR
 */
bootloader_status_t platform_get_unlock_details(unlock_block_t *unlock_block_out)
{
	bootloader_status_t ret = BOOTLOADER_OK;
	int err = read_nand_partition_from_type(UNFD_PART_CUST1 | UNFD_HIDDEN_PART,
	                                        sizeof(unlock_block_t),
	                                        (u_char*)unlock_block_out, 0);

	if (err)
		ret = BOOTLOADER_READ_ERROR;

	return ret;
}

/*
 * Write unlock block to storage location
 * @return BOOTLOADER_OK if successfully stored otherwise BOOTLOADER_WRITE_ERROR
 */
bootloader_status_t platform_write_unlock_details(const unlock_block_t *unlock_block)
{
	bootloader_status_t ret = BOOTLOADER_OK;
	struct unlock_meta ul_data;
	/* Read the pre-existing data */
	int err = read_nand_partition_from_type(UNFD_PART_CUST1 | UNFD_HIDDEN_PART,
	                                        sizeof(struct unlock_meta),
	                                        (u_char*)&ul_data, 0);

	if (err) {
		ret = BOOTLOADER_READ_ERROR;
		goto out;
	}

	/* Replace the unlock block */
	memcpy(&ul_data.unlock_block, unlock_block, sizeof(unlock_block_t));

	/* write back the unlock metadata */
	err = erase_and_write_nand_partition_to_type(
	                                 UNFD_PART_CUST1 | UNFD_HIDDEN_PART,
	                                 sizeof(struct unlock_meta),
	                                 (u_char*)&ul_data, 0);

	if (err)
		ret = BOOTLOADER_WRITE_ERROR;

out:
	return ret;
}

/*
 * Cause the device to appear locked
 * This can just write zeros to the stored unlock_block, but it may require
 * additional steps per platform
 * @return BOOTLOADER_OK if successfully stored otherwise BOOTLOADER_WRITE_ERROR
 */
bootloader_status_t platform_relock(void)
{
	bootloader_status_t ret = BOOTLOADER_OK;
	/* Destroy all the unlock data, as well as the factory test mode data */
	static struct unlock_meta zero_unlock_data = {0};

	int err = erase_and_write_nand_partition_to_type(
	                                 UNFD_PART_CUST1 | UNFD_HIDDEN_PART,
	                                 sizeof(struct unlock_meta),
	                                 (u_char*)&zero_unlock_data, 0);

	/* Force state re-evaluation on next query */
	platform_set_unlock_state(UNLOCK_STATE_INVALID);

	if (err)
		ret = BOOTLOADER_WRITE_ERROR;

	return ret;
}
/*
 * Verify cryptographic signature
 * e.g. using RSASSA-PSS 2048bit or ECDSA 224bit
 *
 * Use hardware acceleration if available
 *
 * @return BOOTLOADER_VERIFY_SUCCESS if signature verifies correctly, otherwise BOOTLOADER_VERIFY_FAILURE
 */
bootloader_status_t platform_verify_signature( const hash_t *hashed_challenge,
                                               const signature_t *signature,
                                               key_id_t public_key )
{
	bootloader_status_t ret = BOOTLOADER_VERIFY_FAILURE;
	u8* unlock_key = (u8*)KEY_CUST_LOAD_ADDRESS;
	int key_page = 0;
	int key_offset = 0;

	switch (public_key) {
		case UNRESTRICTED_UNLOCK_PUBLIC_KEY:
			key_page = PROD_KEY_PAGE;
			key_offset = FULL_UNLOCK_KEY_OFFSET;
			break;
		case BOOT_COUNTED_UNLOCK_PUBLIC_KEY:
			key_page = PROD_KEY_PAGE;
			key_offset = MFG_UNLOCK_KEY_OFFSET;
			break;
		default:
			printf("Unsupported key type requested (%d)\n", public_key);
			goto out;
	}

	load_ring_public_key(unlock_key, key_page, key_offset, RSA_KEYSIZE);

	if (runRSA((U32)(signature->value), (U32 *)(hashed_challenge->value), (U32 *)unlock_key)) {
		ret = BOOTLOADER_VERIFY_SUCCESS;
	}
out:
	return ret;
}

/*
 * Fetches the current restricted boot count
 * Boots are only counted whilst restricted.
 *
 * If not implemented - just set count=0 and return BOOTLOADER_OK
 *
 * @return BOOTLOADER_OK if count retrieved successfully, otherwise BOOTLOADER_READ_ERROR
 */
bootloader_status_t platform_get_current_boot_count(uint32_t *count)
{
	bootloader_status_t ret = BOOTLOADER_OK;
	struct unlock_meta ul_data;
	int err = read_nand_partition_from_type(UNFD_PART_CUST1 | UNFD_HIDDEN_PART,
	                                        sizeof(struct unlock_meta),
	                                        (u_char*)&ul_data, 0);

	if (err)
		ret = BOOTLOADER_READ_ERROR;
	else
		*count = ul_data.boot_counter;

	return ret;
}

/*
 * Increments the current restricted boot count in non-volatile memory
 * If not implemented - just do nothing and return BOOTLOADER_OK
 *
 * @return BOOTLOADER_OK if count incremented successfully, otherwise BOOTLOADER_WRITE_ERROR
 */
bootloader_status_t platform_increment_boot_count(void)
{
	bootloader_status_t ret = BOOTLOADER_OK;
	unlock_state_t unlock_state;
	struct unlock_meta ul_data;
	int err;

	ret = platform_get_unlock_state(&unlock_state);
	if (ret != BOOTLOADER_OK) {
		printf("%s: Error retrieving unlock state.  Relocking\n", __func__);
		platform_relock();
		goto out;
	}
	if (unlock_state >= UNLOCK_STATE_FULL_UNLOCK) {
		/* Don't need to increment the unlock count for unrestricted devices,
		   so let's not waste flash write cycles */
		goto out;
	}

	err = read_nand_partition_from_type(UNFD_PART_CUST1 | UNFD_HIDDEN_PART,
	                                        sizeof(struct unlock_meta),
	                                        (u_char*)&ul_data, 0);

	if (err) {
		ret = BOOTLOADER_READ_ERROR;
		goto out;
	}

	ul_data.boot_counter++;

	err = erase_and_write_nand_partition_to_type(
	                                 UNFD_PART_CUST1 | UNFD_HIDDEN_PART,
	                                 sizeof(struct unlock_meta),
	                                 (u_char*)&ul_data, 0);

	if (err) {
		ret = BOOTLOADER_WRITE_ERROR;
		goto out;
	}

out:
	return ret;
}


bootloader_status_t platform_set_factory_test_mode(bool enabled, uint32_t factory_test_port)
{
	bootloader_status_t ret = BOOTLOADER_OK;
	struct unlock_meta ul_data;
	int err;

	err = read_nand_partition_from_type(UNFD_PART_CUST1 | UNFD_HIDDEN_PART,
	                                        sizeof(struct unlock_meta),
	                                        (u_char*)&ul_data, 0);
	if (err) {
		ret = BOOTLOADER_READ_ERROR;
		goto out;
	}

	if (enabled) {
		ul_data.factory_test.enabled = FACTORY_TEST_ENABLED;
		ul_data.factory_test.port = factory_test_port;
	} else {
		memset(&ul_data.factory_test, 0, sizeof(factory_test_data_t));
	}

	err = erase_and_write_nand_partition_to_type(
	                                 UNFD_PART_CUST1 | UNFD_HIDDEN_PART,
	                                 sizeof(struct unlock_meta),
	                                 (u_char*)&ul_data, 0);

	if (err) {
		ret = BOOTLOADER_WRITE_ERROR;
		goto out;
	}

out:
	return ret;
}

bootloader_status_t platform_get_factory_test_mode(bool *enabled, uint32_t *factory_test_port)
{
	bootloader_status_t ret = BOOTLOADER_OK;
	struct unlock_meta ul_data;
	int err;

	err = read_nand_partition_from_type(UNFD_PART_CUST1 | UNFD_HIDDEN_PART,
	                                        sizeof(struct unlock_meta),
	                                        (u_char*)&ul_data, 0);
	if (err) {
		ret = BOOTLOADER_READ_ERROR;
		goto out;
	}

	if (enabled != NULL) {
		*enabled = ul_data.factory_test.enabled == FACTORY_TEST_ENABLED ? true : false;
	}
	if (factory_test_port != NULL) {
		*factory_test_port = ul_data.factory_test.port;
	}

out:
	return ret;
}


/*
 * Retrieves the device specific data
 *
 * This should include Device-Serial-Number, MAC address, flash memory unique ID, etc.
 *
 * It expected that the platform unique values are hashed together, although this
 * is not mandatory.  If the data is not hashed, the full length of the returned data
 * must be initialised with constant data or the challenge will not match
 *
 * @return BOOTLOADER_OK if retrieved successfully, otherwise BOOTLOADER_READ_ERROR
 */
bootloader_status_t platform_get_device_unique_data( unique_data_t *unique_data )
{
	int randaddr;
	bootloader_status_t ret = BOOTLOADER_READ_ERROR;
	static const int page_size = 2048;
	u8* loadaddr = (u8*)KEY_CUST_LOAD_ADDRESS;
	int err = read_otp_page(0, loadaddr, page_size, 0);
	if (err)
		goto out;
	do
	{ /* The first 44 bytes are not randomised */
		randaddr = (rand() & 0x1FF) << 2;
	} while (randaddr < sizeof(struct otp_meta));

	/* Page 0 of the nand flash contains device random data after the
	   platform metadata.  feed this into the random number seed */
	srand(*((uint32_t*)(loadaddr + randaddr)) ^ rand());

	ret = platform_hash((hash_t *)unique_data, loadaddr, page_size);
	if (ret != BOOTLOADER_OK)
		goto out;

	if (!dsn_valid)
	{
		memcpy(dsn.value, ((struct otp_meta *)loadaddr)->dsn, DSN_SIZE);
		dsn_valid = true;
	}
out:
	return ret;
}

bootloader_status_t platform_get_device_unique_tag( unique_tag_t *unique_tag )
{
	int err;
	bootloader_status_t ret = BOOTLOADER_READ_ERROR;
	if (!dsn_valid)
	{
		static const int page_size = 2048;
		u8* loadaddr = (u8*)KEY_CUST_LOAD_ADDRESS;
		err = read_otp_page(0, loadaddr, page_size, 0);
		if (err)
			goto out;
		memcpy(dsn.value, ((struct otp_meta *)loadaddr)->dsn, DSN_SIZE);
		dsn_valid = true;
	}
	memcpy(unique_tag, dsn.value, sizeof(unique_tag_t));
	ret = BOOTLOADER_OK;
out:
	return ret;
}

/*
 * Cryptographically hash data
 * e.g. SHA-256
 * Use hardware acceleration where possible
 *
 * @return BOOTLOADER_OK if retrieved successfully, otherwise BOOTLOADER_HASH_FAILURE
 */
bootloader_status_t platform_hash( hash_t *hash_out, const void* data, uint32_t data_size )
{
	uint32_t dataaligned;
	uint8_t *allocptr = NULL;
	if ((uint32_t)data & 0xF)
	{ /* We must be 16byte aligned */
		int alloc_size = data_size + 0xF;
		boot_debug("%s: Input data must be 16 byte aligned.  Copying...\n", __func__);
		allocptr = malloc(alloc_size);
		if (!allocptr)
		{
			printf("%s: Failed to allocate %d bytes\n",
			       __func__, alloc_size);
			return BOOTLOADER_OTHER_ERROR;
		}
		dataaligned = ((uint32_t)allocptr + 0xF) & (~0xF);
		memcpy((uint8_t *)dataaligned, data, data_size);
		boot_debug("%s: Copied %d bytes to aligned pointer 0x%x\n",
		           __func__, data_size, dataaligned);
	}
	else
	{
		dataaligned = (uint32_t)data;
	}
	MDrv_SHA_Run(dataaligned, data_size, E_SHA_MODE_256, (U16*)hash_out->value);
	if (allocptr)
	{
		free(allocptr);
	}
	return BOOTLOADER_OK;
}


/*
 * Generate a random number
 *
 * Use hardware RNG if available.
 * Otherwise, part of seed must be stored in non-volatile memory.
 */
bootloader_status_t platform_random(pseudo_uint64_t *rval)
{
	if (!rval)
		return BOOTLOADER_INVALID_PARAMETER;

#if (!defined USE_STATIC_UNLOCK) || (USE_STATIC_UNLOCK == 0)
	/* Normal unlock: return a boot-unique random portion */
	srand(get_ms_ticks() ^ rand());
	rval->value[0] = rand();
	srand(get_ms_ticks() ^ rand());
	rval->value[1] = rand();
	return BOOTLOADER_OK;
#else
	static const int page_size = 2048;
	/* Static unlock: return a device-unique random portion */
	u8* loadaddr = (u8*)KEY_CUST_LOAD_ADDRESS;
	int err = read_otp_page(0, loadaddr, page_size, 0);
	if (err)
		goto out;

	loadaddr += sizeof(struct otp_meta);
	/* Page 0 OTP data for each device is randomly generated at provision time, so this
	 * data is static per device, but random across devices meaning unlock codes are not
	 * transferrable */
	memcpy(&(rval->value[0]), loadaddr, sizeof(rval->value[0]));
	memcpy(&(rval->value[1]), loadaddr + sizeof(rval->value[0]), sizeof(rval->value[1]));

	return BOOTLOADER_OK;
out:
	return BOOTLOADER_READ_ERROR;
#endif
}

/*
 * Reboot system
 * Must not return
 */
void platform_reboot(void)
{
	do_reset(NULL, 0, 0, NULL);
}
